/*
 * routines for loading the fabric description from a file
 */
#include <stdio.h>

#include "libfma.h"
#include "lf_db.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_product_def.h"
#include "lf_switch.h"

/*
 * create a new host and add it to a fabric
 */
struct lf_host *
lf_add_host_to_fabric(
  struct lf_fabric *fp,
  char *hostname)
{
  struct lf_host *hp;
  int rc;

  /* alloacate a host and fill in hostname */
  LF_CALLOC(hp, struct lf_host, 1);
  LF_DUP_STRING(hp->hostname, hostname);

  /* add this host into the fabric */
  rc = lf_add_existing_host_to_fabric(fp, hp);
  if (rc != 0) LF_ERROR(("Error adding  host to fabric"));

  return hp;

 except:
  lf_free_host(hp);
  return NULL;
}

/*
 * Add an existing host to a fabric
 */
int
lf_add_existing_host_to_fabric(
  struct lf_fabric *fp,
  struct lf_host *hp)
{
  void *nha;
  int nnh;

  /* make a place for the new host */
  nnh = fp->num_hosts + 1;
  nha = realloc(fp->hosts, nnh * sizeof(struct lf_host *));
  if (nha == NULL) LF_ERROR(("Error growing fabric host array"));

  /* put this host in the fabric hosts array */
  fp->hosts = nha;
  fp->hosts[fp->num_hosts] = hp;
  fp->num_hosts = nnh;

  return 0;

 except:
  return -1;
}

/*
 * Add a NIC to a host struct
 */
struct lf_nic *
lf_add_nic_by_product_id(
  struct lf_host *hp,
  char *product_id)
{
  struct lf_nic *nicp;
  int rc;

  /* allocate new NIC from product ID */
  nicp = lf_alloc_nic_by_product_id(product_id);
  if (nicp == NULL) LF_ERROR(("Error allocating new NIC"));

  /* Add this NIC to the host */
  rc = lf_add_existing_nic_to_host(nicp, hp);
  if (rc != 0) LF_ERROR(("Error adding new NIC to host"));

  return nicp;

 except:
  lf_free_nic(nicp);
  return NULL;
}

/*
 * Add an existing NIC to a host, expanding the host nic array
 */
int
lf_add_existing_nic_to_host(
  struct lf_nic *nicp,
  struct lf_host *hp)
{
  void *nnp;
  int nnn;

  /* link new nic to owning host */
  nicp->slot = hp->num_nics;
  nicp->host = hp;

  /* expand nic array in host struct */
  nnn = hp->num_nics+1;
  nnp = realloc(hp->nics, nnn * sizeof(struct lf_nic *));
  if (nnp == NULL) LF_ERROR(("errro re-allocating host NIC array"));

  /* update host struct */
  hp->nics = nnp;
  hp->nics[hp->num_nics] = nicp;
  hp->num_nics = nnn;

  return 0;

 except:
  return -1;
}

/*
 * remove a NIC from its owning host and from the fabric altogether
 */
void
lf_remove_nic_from_host(
  struct lf_nic *nicp)
{
  struct lf_host *hp;
  int n;

  /* adjust host array of NICs (no need to realloc) */
  hp = nicp->host;
  --hp->num_nics;
  for (n=nicp->slot; n<hp->num_nics; ++n) {
    hp->nics[n] = hp->nics[n+1];
    hp->nics[n]->slot = n;
  }
}

/*
 * Remove a host from a fabric
 */
void
lf_remove_host_from_fabric(
  struct lf_fabric *fp,
  struct lf_host *hp)
{
  int h;

  /* find the slot with this host */
  for (h=0; h<fp->num_hosts; ++h) {
    if (fp->hosts[h] == hp) break;
  }

  /* if not found, just return */
  if (h >= fp->num_hosts) return;

  /* shift the rest of the hosts down one */
  --fp->num_hosts;
  while (h < fp->num_hosts) {
    fp->hosts[h] = fp->hosts[h+1];
    ++h;
  }
}

/*
 * Remove a link from a node in the fabric.  This removes the link information
 * from both ends of the link.
 */
void
lf_remove_link_by_topo(
  union lf_node *np,
  int port)
{
  union lf_node *onp;
  int oport;
  union lf_node *pnp;
  int pport;

  /* find other side of the link.  If none, all done! */
  onp = lf_follow_topo_link(np, port, &oport);
  if (onp == NULL) return;

  /* link both ends to nothing */
  lf_make_topo_link(np, port, NULL, 0);
  lf_make_topo_link(onp, oport, NULL, 0);

  /* link the phys port for each to nothing */
  pnp = lf_phys_for_topo(np, port, &pport);
  lf_make_phys_link(pnp, pport, NULL, 0);

  pnp = lf_phys_for_topo(onp, oport, &pport);
  lf_make_phys_link(pnp, pport, NULL, 0);
}

/*
 * Add a link between two nodes.  The nodes specified are topologically
 * linked, and then the appropriate physical links are made as well.
 */
void
lf_connect_topo_nodes(
  union lf_node *np1,
  int port1,
  union lf_node *np2,
  int port2)
{
  union lf_node *physnode1;
  int physport1;
  union lf_node *physnode2;
  int physport2;

  /* first, the easy part - make the topo links */
  lf_make_topo_link(np1, port1, np2, port2);
  lf_make_topo_link(np2, port2, np1, port1);

  /* find the appropriate physical port for each node/port */
  physnode1 = lf_phys_for_topo(np1, port1, &physport1);
  physnode2 = lf_phys_for_topo(np2, port2, &physport2);

  lf_make_phys_link(physnode1, physport1, physnode2, physport2);
  lf_make_phys_link(physnode2, physport2, physnode1, physport1);
}

/*
 * find the physical node that fronts a given topological node
 */
union lf_node *
lf_phys_for_topo(
  union lf_node *tnp,
  int tport,
  int *pportp)
{
  union lf_node *pnp;
  int pport;

  if (tnp->ln_type == LF_NODE_NIC) {
    struct lf_nic *nicp;

    nicp = LF_NIC(tnp);

    /* if we have an xcvr, it's the phys port, else we are our own phys port */
    pnp = nicp->phys_ports[tport];
    if (pnp != NULL && pnp->ln_type == LF_NODE_NIC_XCVR) {
      pport = nicp->phys_rports[tport] - LF_XCVR(pnp)->num_conns;
    } else {
      pnp = tnp;
      pport = tport;
    }

  } else if (tnp->ln_type == LF_NODE_XBAR) {
    struct lf_xbar *xp;

    xp = LF_XBAR(tnp);

    /* get the physical node - if not LC xcvr, we front ourselves */
    pnp = xp->phys_ports[tport];
    if (pnp != NULL && pnp->ln_type == LF_NODE_LC_XCVR) {
      pport = xp->phys_rports[tport] - LF_XCVR(pnp)->num_conns;
if (pport == -1) {
  printf("bad pport! \n");
  abort();
}
    } else {
      pnp = tnp;
      pport = tport;
    }

  } else {
    LF_ERROR(("Bad node type, not topo!"));
  }

  /* return what we found */
  if (pportp != NULL) *pportp = pport;
  return pnp;

 except:
  return NULL;
}

/*
 * Add an existing enclosure to a fabric
 */
int
lf_add_existing_enclosure_to_fabric(
  struct lf_fabric *fp,
  struct lf_enclosure *ep)
{
  void *nea;
  int nne;

  /* make a place for the new enclosure */
  nne = fp->num_enclosures + 1;
  nea = realloc(fp->enclosures, nne * sizeof(struct lf_enclosure *));
  if (nea == NULL) LF_ERROR(("Error growing fabric enclosure array"));

  /* put this enclosure in the fabric enclosures array */
  fp->enclosures = nea;
  fp->enclosures[fp->num_enclosures] = ep;
  fp->num_enclosures = nne;

  return 0;

 except:
  return -1;
}

/*
 * Remove an enclosure from the fabric
 */
void
lf_remove_enclosure_from_fabric(
  struct lf_fabric *fp,
  struct lf_enclosure *ep)
{
  int e;

  /* find the slot with this enclosure */
  for (e=0; e<fp->num_enclosures; ++e) {
    if (fp->enclosures[e] == ep) break;
  }

  /* if not found, just return */
  if (e >= fp->num_enclosures) return;

  /* shift the rest of the enclosures down one */
  --fp->num_enclosures;
  while (e < fp->num_enclosures) {
    fp->enclosures[e] = fp->enclosures[e+1];
    ++e;
  }
}
